"""The starting up screen"""

import random
import math
import pygame
import os
import copy

import serge.actor
import serge.visual
import serge.events
import serge.common
import serge.sound
import serge.blocks.utils
import serge.blocks.visualblocks
import serge.blocks.behaviours
import serge.blocks.actors

if serge.common.PYMUNK_OK:
    import pymunk
    import serge.physical

from theme import G, theme
import common 
import mainmenu
import mainscreen

class StartScreen(serge.blocks.actors.ScreenActor):
    """The logic for the start screen"""
    
    def __init__(self, options):
        """Initialise the screen"""
        super(StartScreen, self).__init__('item', 'start-screen')
        self.options = options
        
    def addedToWorld(self, world):
        """Added to the world"""
        super(StartScreen, self).addedToWorld(world)
        #
        # The behaviour manager
        self.manager = serge.blocks.behaviours.BehaviourManager('behaviours', 'behaviours')
        self.manager.assignBehaviour(None, serge.blocks.behaviours.KeyboardQuit(), 'keyboard-quit')
        self.world.addActor(self.manager)
        #
        # Logo
        self.logo = serge.blocks.utils.addSpriteActorToWorld(world, 'logo', 'logo', 
            'logo', 'ui', center_position=G('logo-position'))
        self.bg = serge.blocks.utils.addSpriteActorToWorld(world, 'bg', 'bg', 
            'start-bg', 'background', center_position=G('start-bg-position'))
        #
        # The grid to show different levels
        self.grid = serge.blocks.utils.addActorToWorld(world, 
            serge.blocks.layout.Grid('grid', 'grid', **G('level-grid')),
            layer_name='ui', center_position=G('level-grid-center-position'))
        #
        # Add individual levels
        w = G('level-grid')['size'][0]
        self.locks = []
        for idx in range(1, 100):
            theme_name = 'level-%d' % idx
            if not theme.hasTheme(theme_name):
                break
            #
            # Get graphic
            sprite_name = G('background', theme_name)
            if not sprite_name:            
                sprite_name = 'default-bg'
            #
            # Add button
            button = serge.actor.MountableActor('button', 'button-%d' % idx)
            button.visual = serge.blocks.visualblocks.SpriteText(
                '' if G('locked', theme_name) else G('name', theme_name), 
                sprite_name=sprite_name, **G('level-button-text'))
            button.setZoom(G('level-grid-bg-zoom'))
            self.grid.addActor(((idx-1) % w, (idx-1) // w), button)
            button.linkEvent(serge.events.E_LEFT_CLICK, self.levelButtonClick, idx)
            #
            highlight = serge.actor.Actor('highlight', 'highlight-%d' % idx)
            highlight.visual = serge.blocks.visualblocks.Rectangle(G('level-highlight-size'), **G('level-highlight'))
            button.mountActor(highlight, (0, 0))
            highlight.setLayerName('ui-background')
            highlight.active = not G('locked', theme_name)
            #
            # Lock
            lock = serge.actor.Actor('lock', 'lock-%d' % idx)
            lock.setSpriteName('lock')
            button.mountActor(lock, (0, 0))
            lock.setLayerName('front-ui')
            lock.visible = G('locked', theme_name)
            #
            # Check mark
            check = serge.actor.Actor('check', 'check-%d' % idx)
            check.setSpriteName('check-mark')
            button.mountActor(check, (0, 0))
            check.setLayerName('front-ui')
            check.visible = G('completed', theme_name)            
            #
            self.locks.append(G('locked', theme_name))
        #
        serge.blocks.utils.addVisualActorToWorld(world, 'version', 'version',
            serge.visual.Text('v%s' % common.version, **G('version-text')), 'ui',
            center_position=G('version-text-position'))
        #
        credits = serge.blocks.utils.addVisualActorToWorld(world, 'credits', 'credits',
            serge.visual.Text('Credits', **G('credits-text')), 'ui',
            center_position=G('credits-text-position'))
        credits.linkEvent(serge.events.E_LEFT_CLICK, self.creditsClick)
        #
        achievements = serge.blocks.utils.addVisualActorToWorld(world, 'achievements', 'achievements',
            serge.visual.Text('Achievements', **G('achievements-text')), 'ui',
            center_position=G('achievements-text-position'))
        achievements.linkEvent(serge.events.E_LEFT_CLICK, self.achievementsClick)
        #
        # Modes
        mode_bar = serge.blocks.utils.addActorToWorld(world,
            serge.blocks.layout.HorizontalBar('hbar', 'hbar', width=G('modes-bar-width'), height=G('modes-bar-height'),
                background_colour=G('modes-bar-background-colour')),
            layer_name='ui', center_position=G('modes-bar-position'))
        #
        mode_bar.addActor(serge.blocks.actors.StringText('modes', 'modes', 
            'Game Mode', font_size=G('mode-label-font-size'), colour=G('mode-label-font-colour')))
        #
        self.modes = mode_bar.addActor(
            serge.blocks.actors.ToggledMenu('mode-menu', 'mode-menu',
                items=['Normal', 'Reverse'],
                layout=serge.blocks.layout.HorizontalBar('modes', 'modes', width=G('modes-width')),
                default='Normal', **G('modes-style')))
        self.modes.layout.setLayerName('front-ui')
        #
        mode_bar.addActor(serge.actor.Actor('dummy', 'dummy')) # Spacer
        #
        self.world.linkEvent(serge.events.E_ACTIVATE_WORLD, self.activateWorld)
        self.activateWorld(None, None)
        self.playlist = ['track-1', 'track-2', 'track-3'] if not serge.sound.Music.playlist else serge.sound.Music.playlist
        #
        # Nice Phaser to make a cool effect on the background
        self.phaser = serge.blocks.effects.ColourPhaser(G('red-phaser'), G('green-phaser'), G('blue-phaser'))
        self.world.addActor(self.phaser)
        
    def levelButtonClick(self, obj, level):
        """Clicked a level button"""
        if self.options.cheat or not self.locks[level-1]:
            self.log.info('Clicked on level %d' % level)
            mainscreen.main(self.options, level, self.modes.getSelectionIndex())
            self.engine.setCurrentWorldByName('main-screen')
            serge.sound.Music.setPlaylist(self.playlist)

    def updateActor(self, interval, world):
        """Update the menu"""
        super(StartScreen, self).updateActor(interval, world)
        #
        # Reset colouring
        moused_over = self.mouse.getActorsUnderMouse(world).findActorsByTag('highlight')
        for highlight in self.world.findActorsByTag('highlight'):
            highlight.visible = False
        if moused_over:
            moused_over[0].visible = True

    def creditsClick(self, obj, arg):
        """Clicked on the credits"""
        self.engine.setCurrentWorldByName('credit-screen')

    def achievementsClick(self, obj, arg):
        """Clicked on the achievements"""
        self.engine.setCurrentWorldByName('achievements-screen')
        
    def unlockLevel(self, level):
        """Unlock a certain level"""
        self.world.findActorByName('button-%d' % level).visual.text_visual.setText(G('name', 'level-%d' % level))
        self.world.findActorByName('highlight-%d' % level).active = True
        self.world.findActorByName('lock-%d' % level).visible = False
        self.locks[level-1] = False

    def completeLevel(self, level):
        """Complete a certain level"""
        self.world.findActorByName('button-%d' % level).visual.text_visual.setText(G('name', 'level-%d' % level))
        self.world.findActorByName('check-%d' % level).visible = True

    def activateWorld(self, obj, arg):
        """The world is activated"""
        if not serge.sound.Music.isPlayingSong('intro'):
            serge.sound.Music.play('intro')        
                
def main(options):
    """Create the main logic"""
    #
    # The screen actor
    s = StartScreen(options)
    world = serge.engine.CurrentEngine().getWorld('start-screen')
    world.addActor(s)
    #
    # Screenshots
    if options.screenshot:
        manager = world.findActorByName('behaviours')
        manager.assignBehaviour(None, 
            serge.blocks.behaviours.SnapshotOnKey(key=pygame.K_s, size=G('screenshot-size')
                , overwrite=False, location='screenshots'), 'screenshots')

    #
    return s
        

